package com.seatel.mobilehall.ui.home.viewholder

import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import androidx.recyclerview.widget.LinearLayoutManager
import com.bumptech.glide.Glide
import com.seatel.mobilehall.databinding.LayoutBuySimSummaryItemBinding
import com.seatel.mobilehall.databinding.LayoutContentPlanSubscriptionBinding
import com.seatel.mobilehall.databinding.LayoutContentSpecialOfferBinding
import com.seatel.mobilehall.databinding.LayoutContentSpecialOfferItemBinding
import com.seatel.mobilehall.databinding.LayoutContentTopUpActiveBinding
import com.seatel.mobilehall.ui.base.adapter.BaseRecyclerAdapter
import com.seatel.mobilehall.ui.home.model.BuySimSummaryItemModel
import com.seatel.mobilehall.util.AppUtils
import com.seatel.mobilehall.util.Constant
import com.seatel.mobilehall.util.SeatelUtils

class BuySimSummaryHolder(private val binding: LayoutBuySimSummaryItemBinding) :
    BaseViewHolder<BuySimSummaryItemModel>(binding.root) {

    override fun binData(baseRecyclerAdapter: BaseRecyclerAdapter<BuySimSummaryItemModel>) {
        val dataModel = baseRecyclerAdapter.getItem(adapterPosition)
        binding.textTitle.text = dataModel.title
        getViewByType(dataModel)
    }

    private fun getViewByType(data: BuySimSummaryItemModel) {
        binding.layoutChildContent.removeAllViews()
        when (data) {
            is BuySimSummaryItemModel.SpecialOffer -> {
                addTopSpecialOfferView(data)
            }

            is BuySimSummaryItemModel.TopUpAmount -> {
                addTopUpAmountView(data)
            }

            is BuySimSummaryItemModel.PlanSubscription -> {
                addPlanSubscriptionView(data)
            }
        }
    }

    private fun addTopSpecialOfferView(data: BuySimSummaryItemModel.SpecialOffer) {
        val specialOfferBinding = LayoutContentSpecialOfferBinding.inflate(
            LayoutInflater.from(binding.root.context), binding.layoutChildContent, false
        )
        val mRecyclerView = specialOfferBinding.recyclerSpecialOffer
        mRecyclerView.setHasFixedSize(true)
        mRecyclerView.layoutManager = object : LinearLayoutManager(binding.root.context) {
            override fun canScrollVertically(): Boolean = false
        }
        if (data.data != null) {
            val specialOfferAdapter = SpecialOfferAdapter(data.data!!)
            mRecyclerView.adapter = specialOfferAdapter
        }
        binding.layoutChildContent.addView(specialOfferBinding.root)
    }

    private fun addTopUpAmountView(data: BuySimSummaryItemModel.TopUpAmount) {
        val topUpBinding = LayoutContentTopUpActiveBinding.inflate(
            LayoutInflater.from(binding.root.context), binding.layoutChildContent, false
        )
        topUpBinding.textAmount.text = AppUtils.decimalFormat().format(data.amount)
        binding.layoutChildContent.addView(topUpBinding.root)
    }

    private fun addPlanSubscriptionView(data: BuySimSummaryItemModel.PlanSubscription) {
        val planSubscriptionBinding = LayoutContentPlanSubscriptionBinding.inflate(
            LayoutInflater.from(binding.root.context), binding.layoutChildContent, false
        )
        planSubscriptionBinding.textPlanName.text = data.name
        val dayStr = SeatelUtils.translatForDialog(binding.root.context, "Day")

        if (data.isAddOn && data.addOnPrice > 0.0) {
            planSubscriptionBinding.textDefault.visibility = View.VISIBLE
            var str = "(${
                checkDayStr(
                    dayStr,
                    data.duration!!
                )
            } = ${SeatelUtils.formatDecimalNumber(data.addOnPrice, "$#.##")})"
            if (data.code == Constant.VALUE_PACK) {
                str = "(${
                    SeatelUtils.formatDecimalNumber(
                        data.addOnPrice,
                        "$#.#"
                    )
                } = ${SeatelUtils.formatDecimalNumber(data.duration!!.toDouble(), "$###,###")})"
            }
            planSubscriptionBinding.textDefault.text = str
        } else {
            planSubscriptionBinding.textDefault.visibility = View.INVISIBLE
        }

        val imageIcon = planSubscriptionBinding.imagePlanIcon
        if (data.planIcon!!.isNotEmpty()) {
            Glide.with(binding.root.context).load(data.planIcon).into(imageIcon)
        } else if (data.icon != null) {
            imageIcon.setImageResource(data.icon!!)
        }

        binding.layoutChildContent.addView(planSubscriptionBinding.root)
    }

    private fun checkDayStr(strDay: String, dayCount: Int): String {
        return when (SeatelUtils.getLangaugeCode(binding.root.context)) {
            Constant.KHMER, Constant.CHINESE -> "$dayCount$strDay"
            else -> "$dayCount-$strDay"
        }
    }

    private class SpecialOfferAdapter(mData: List<String>) : BaseRecyclerAdapter<String>(mData) {
        override fun onCreateViewHolder(parent: ViewGroup, viewType: Int): BaseViewHolder<String> {
            val binding = LayoutContentSpecialOfferItemBinding.inflate(
                LayoutInflater.from(parent.context), parent, false
            )
            return SpecialOfferHolder(binding)
        }
    }

    private class SpecialOfferHolder(private val binding: LayoutContentSpecialOfferItemBinding) :
        BaseViewHolder<String>(binding.root) {

        override fun binData(baseRecyclerAdapter: BaseRecyclerAdapter<String>) {
            val data = baseRecyclerAdapter.getItem(adapterPosition)
            binding.textDescription.text = data
        }
    }
}
